<?php
namespace Nitrox\EventBridge\Setup;

use Magento\Framework\Setup\UpgradeDataInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\ModuleContextInterface;

use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Integration\Api\OauthServiceInterface;
use Magento\Integration\Api\AuthorizationServiceInterface;

use Magento\Framework\HTTP\Client\Curl;
use Psr\Log\LoggerInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class UpgradeData implements UpgradeDataInterface
{
    // Define constant for org token path
    private const ORG_TOKEN_PATH = 'admin/nitrox/nitrox_org_token';

    // Declare class properties
    protected $integrationService;
    protected $oauthService;
    protected $authService;
    protected $curl;
    protected $logger;
    protected $scopeConfig;
    protected $storeManager;

    // Constructor to initialize class properties
    public function __construct(
        IntegrationServiceInterface $integrationService,
        OauthServiceInterface $oauthService,
        AuthorizationServiceInterface $authService,
        Curl $curl,
        LoggerInterface $logger,
        ScopeConfigInterface $scopeConfig,
        StoreManagerInterface $storeManager
    ) {
        $this->integrationService = $integrationService;
        $this->oauthService       = $oauthService;
        $this->authService        = $authService;
        $this->curl               = $curl;
        $this->logger             = $logger;
        $this->scopeConfig        = $scopeConfig;
        $this->storeManager       = $storeManager;
    }

    // The upgrade method called during setup
    public function upgrade(ModuleDataSetupInterface $setup, ModuleContextInterface $context)
    {
        $this->logger->info("[Nitrox_EventBridge] UpgradeData triggered.");

        // Get org token from DB or config
        $orgToken = $this->fetchOrgTokenFromDb($setup);
        if (!$orgToken) {
            $orgToken = (string)$this->scopeConfig->getValue(self::ORG_TOKEN_PATH, ScopeInterface::SCOPE_STORE);
        }
        $this->logger->info("[Nitrox_EventBridge] org_token: " . ($orgToken ?: 'NOT SET'));

        // Get Store Base URL
        $baseUrl = $this->storeManager->getStore()->getBaseUrl();
        $this->logger->info("[Nitrox_EventBridge] Store Base URL: " . $baseUrl);

        try {
            $integrationName = 'nitro_magento_sso';

            // Create or load integration
            $integration = $this->integrationService->findByName($integrationName);
            if (!$integration->getId()) {
                // Create new integration if not found
                $integration = $this->integrationService->create([
                    'name'       => $integrationName,
                    'status'     => 1,
                    'setup_type' => 0
                ]);
                $this->logger->info("[Nitrox_EventBridge] New integration created.");
            } else {
                $this->logger->info("[Nitrox_EventBridge] Existing integration found (ID: {$integration->getId()}).");
            }

            // Handle consumer creation or loading
            if (!$integration->getConsumerId()) {
                $consumer = $this->oauthService->createConsumer(['name' => $integrationName]);
                $integration->setConsumerId($consumer->getId());
                $integration = $this->integrationService->update($integration);
                $this->logger->info("[Nitrox_EventBridge] Consumer created (ID: {$consumer->getId()}).");
            } else {
                $consumer = $this->oauthService->loadConsumer($integration->getConsumerId());
                $this->logger->info("[Nitrox_EventBridge] Consumer loaded (ID: {$consumer->getId()}).");
            }

            // Grant full customer-related permissions
            $permissions = [
                'Magento_Customer::customer', // General permission for customer data
                'Magento_Customer::manage',   // Manage customer (Create, Update, Delete)
                'Magento_Customer::view',     // View customer data
                'Magento_Customer::create',   // Create customer accounts
                'Magento_Customer::delete'    // Delete customer accounts
            ];

            $this->authService->grantPermissions($integration->getId(), $permissions);

            // Generate access token
            $this->oauthService->createAccessToken($consumer->getId(), true);
            $token = $this->oauthService->getAccessToken($consumer->getId());

            // Prepare payload with integration details and store URL
            $data = [
                'consumer_key'        => $consumer->getKey(),
                'consumer_secret'     => $consumer->getSecret(),
                'access_token'        => $token->getToken(),
                'access_token_secret' => $token->getSecret(),
                'org_token'           => $orgToken ?: null,
                'store_url'           => $baseUrl
            ];

            // Send to microservice (Webhook)
            $url = 'https://svc.nitrocommerce.ai/v1/services/magento/ingest/store_token';
            $this->curl->addHeader("Content-Type", "application/json");
            $this->curl->post($url, json_encode($data));

            // Log status and response from webhook
            $this->logger->info("[Nitrox_EventBridge] CURL Status: " . $this->curl->getStatus());
            $this->logger->info("[Nitrox_EventBridge] CURL Response: " . $this->curl->getBody());
        } catch (\Throwable $e) {
            // Catch and log any errors
            $this->logger->error("[Nitrox_EventBridge] Error: " . $e->getMessage());
        }
    }

    // Fetch org token from DB
    private function fetchOrgTokenFromDb(ModuleDataSetupInterface $setup): ?string
    {
        $connection = $setup->getConnection();
        $table      = $setup->getTable('core_config_data');
        $select     = $connection->select()
            ->from($table, 'value')
            ->where('path = ?', self::ORG_TOKEN_PATH)
            ->order('config_id DESC')
            ->limit(1);
        $value = $connection->fetchOne($select);
        return $value === false ? null : (string)$value;
    }
}
