<?php
namespace Nitrox\EventBridge\Model;

use Nitrox\EventBridge\Api\NitroInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\CustomerFactory;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class Nitro implements NitroInterface
{
    private const SSO_VALIDATION_URL = 'https://t.makehook.ws/jsv1/magento/validate_customer_login';

    private $customerRepository;
    private $customerFactory;
    private $storeManager;
    private $curlClient;
    private $jsonSerializer;
    private $logger;

    public function __construct(
        CustomerRepositoryInterface $customerRepository,
        CustomerFactory $customerFactory,
        StoreManagerInterface $storeManager,
        Curl $curlClient,
        Json $jsonSerializer,
        LoggerInterface $logger
    ) {
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->storeManager = $storeManager;
        $this->curlClient = $curlClient;
        $this->jsonSerializer = $jsonSerializer;
        $this->logger = $logger;
    }

    public function validateToken(string $token): ?array
    {
        try {
            $this->curlClient->setHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]);
            $this->curlClient->setOptions([CURLOPT_FOLLOWLOCATION => true]);

            $url = self::SSO_VALIDATION_URL . '?token=' . urlencode($token);
            $this->curlClient->get($url);

            $responseBody = $this->curlClient->getBody();
            $responseData = $this->jsonSerializer->unserialize($responseBody);

            if (!isset($responseData['status']) || $responseData['status'] !== true) {
                $this->logger->warning('SSO token validation failed: ' . ($responseData['error'] ?? 'Unknown error'));
                return null;
            }

            return $responseData;
        } catch (\Exception $e) {
            $this->logger->error('SSO validation request failed: ' . $e->getMessage(), ['exception' => $e]);
            return null;
        }
    }

    public function getCustomerByEmail(string $email)
    {
        try {
            $websiteId = $this->storeManager->getWebsite()->getId();
            return $this->customerRepository->get($email, $websiteId);
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $this->logger->info('Customer not found by email: ' . $email);
            return null;
        }
    }
}
